/* Copyright (c) 2005 Andrew Choi.  All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are
met:

1. Redistributions of source code must retain the above copyright
   notice, this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright
   notice, this list of conditions and the following disclaimer in the
   documentation and/or other materials provided with the distribution.

3. All advertising materials mentioning features or use of this
   software must display the following acknowledgement:

     This product includes software developed by Andrew Choi.

4. The name "Andrew Choi" may not be used to endorse or promote
   products derived from this software without specific prior written
   permission.

THIS SOFTWARE IS PROVIDED BY ANDREW CHOI "AS IS" AND ANY EXPRESS OR
IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED.  IN NO EVENT SHALL ANDREW CHOI BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
POSSIBILITY OF SUCH DAMAGE.  */

#include "MIDI.h"

#include "SysexReceiveRequest.h"

static void ReadProc(const MIDIPacketList *pktlist, void *readProcRefCon, void *srcConnRefCon)
{
  SysexReceiveRequest *request = (SysexReceiveRequest *)readProcRefCon;

  pthread_mutex_lock(&request->mutex);
  
  const MIDIPacket *packet = &pktlist->packet[0];
  for (int i = 0; i < pktlist->numPackets; i++) {
    CFDataAppendBytes(request->receivedSysexData, (const UInt8 *)packet->data, packet->length);
    packet = MIDIPacketNext(packet);
  }
  
  pthread_mutex_unlock(&request->mutex);
}

bool InitializeSysexReceiveRequest(SysexReceiveRequest *request, MIDIEndpointRef endpointRef)
{
  /* Create mutex before starting receive port.  */
  pthread_mutex_init(&request->mutex, NULL);

  /* Create before starting receive port.  */
  request->receivedSysexData = CFDataCreateMutable(NULL, 0);

  /* Generate unique names for input ports for sysex.  */
  static int seqNum = 0;
  CFStringRef portName = CFStringCreateWithFormat(NULL, NULL, CFSTR("sysex input %d"), seqNum++);

  request->endpointRef = endpointRef;
  request->stopped = false;

  OSStatus s = MIDIInputPortCreate(myMIDIClient, portName, ReadProc, request, &request->portRef);
  CFRelease(portName);
  if (s == noErr)
    s = MIDIPortConnectSource(request->portRef, endpointRef, NULL);
  
  return s == noErr;
}

void TerminateSysexReceiveRequest(SysexReceiveRequest *request)
{
  request->stopped = true;
  (void) MIDIPortDisconnectSource(request->portRef, request->endpointRef);
  (void) MIDIPortDispose(request->portRef);

  /* Destroy mutex after stopping receive port.  */
  pthread_mutex_destroy(&request->mutex);

  /* Release after stopping receive port.  */
  CFRelease(request->receivedSysexData);
}

/* Obviously, caller is responsible for releasing returned data.  */
CFMutableDataRef ReceiveSysexData(SysexReceiveRequest *request)
{
  pthread_mutex_lock(&request->mutex);
  
  CFMutableDataRef result = request->receivedSysexData;
  request->receivedSysexData = CFDataCreateMutable(NULL, 0); 
  
  pthread_mutex_unlock(&request->mutex);
  return result;
}

int NumOfBytesReceived(SysexReceiveRequest *request)
{
  return CFDataGetLength(request->receivedSysexData);
}
